/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/ 
/*
 *  SATA Functions
 *
 */

#include "stdio.h"
#include "ahci.h"
#include "sata.h"

UINT32 Dev28bitLbaAddress; // For 28-bit LBA address
UINT8 PioCmd;              // For identifying IDENTIFY DEVICE Cmd
UINT16 cmdSlot2Use = 0;    // Holds the command list slot to use

SataRegs* sataRegs = SATA_REGS;      // Initialize SATA regs
FirmwareCtrlFeatures swCtrlFeatures;   // Structure to store the HBA capabilities
unsigned char prdTableDataBuff[LISTLENGTH][PRDLENGTH][DATABUFFERLEN]; // Initialize DMA Data Buffer
cmdFis myCmdFis;           // Stores modifiable command list fields

/* Command list header */
#pragma DATA_ALIGN(CmdLists, 1024);  // Command List requires to be 1k Byte Aligned
CmdListHeader CmdLists[32]={0};

/* Command Table */
#pragma DATA_ALIGN(CmdTable, 128);  // Command Table requires 128 Byte Data Alignment
CommandTable CmdTable[LISTLENGTH];

/* Receive FIS */
#pragma DATA_ALIGN(RcvFis, 256);  // Receive FIS requires 256 bytes alignment
ReceiveFis RcvFis;

/*---------------------------------------------------------*
 * SATA Functions                                          *
 *---------------------------------------------------------*/
/* ---------------------------------------------------------------------------- *
 *										*
 *	chceckSysMemorySize( )							*
 *      Checks memory alignment							*
 * ---------------------------------------------------------------------------- */
UINT8 chceckSysMemorySize( ) 
{
	UINT16 i;
	for (i=0; i<LISTLENGTH; i++) 
	{
		if (((UINT32)&CmdTable[i] & 0x7F) != 0)
			break;  // Not 128 byte aligned
	}
	if (i != LISTLENGTH)
		return(1);
	return(0);
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	initMemory( *startAddress, length, seedWord, modifyVal)			*
 *    	Edit memory length long starting from startAddress. Modify to 		*
 *		seedWord and progressively add modifyVal                        * 
 * ---------------------------------------------------------------------------- */
void initMemory(UINT32 *startAddress, UINT32 length, UINT32 seedWord, UINT32 modifyVal)
{
	UINT32 i;
	*startAddress++ = seedWord;
	for (i=0; i<length-1; i++)
	{
		seedWord += modifyVal;
		*startAddress++ = seedWord;
	}
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	clearCmdList( )								*
 *      Initialize command list space						*
 * ---------------------------------------------------------------------------- */
void clearCmdList( ) 
{
    // Clear Host to Device (Command FIS) Space
	initMemory((UINT32*)CmdLists, (LISTLENGTH*(sizeof(CmdListHeader)/4)), 0, 0);
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	clearCmdTables( )							*
 *      Initialize command tables space						*
 * ---------------------------------------------------------------------------- */
void clearCmdTables( ) 
{
	UINT16 cmdSlot;
	for (cmdSlot=0; cmdSlot<LISTLENGTH; cmdSlot++) 
	{
		// Clear Command FIS and ATAPI Command Spaces for Command Header 0
		initMemory((UINT32 *)&CmdTable[cmdSlot], (sizeof(CommandFIS)/4)+(sizeof(Atapi)/4), 0, 0);
		// Clear PRD Descriptor Locations for Command Header 0
		initMemory((UINT32*)((UINT32)&CmdTable[cmdSlot]+0x80), ((sizeof(PRDT)/4)*PRDLENGTH), 0, 0);
	}
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	clearRcvFis( )								*
 *      Initialize Receive FIS space						*
 * ---------------------------------------------------------------------------- */
void clearRcvFis( ) 
{
	// Clear Receive DMA Setup FIS Space.
	initMemory((UINT32*)&RcvFis, (sizeof(DMASetupFis)/4), 0, 0);
	// Clear Receive PIO Setup FIS Space.
	initMemory((UINT32*)((UINT32)&RcvFis+0x20), (sizeof(PIOSetupFis)/4), 0, 0);
	// Clear Receive Device to Host (D2H) Register FIS Space.
	initMemory((UINT32*)((UINT32)&RcvFis+0x40), (sizeof(D2HRegFis)/4), 0, 0);
	// Clear Set Device Bits FIS Space.
	initMemory((UINT32*)((UINT32)&RcvFis+0x58), (sizeof(SetDevBitsFis)/4), 0, 0);
	// Clear Unknown FIS Space.
	initMemory((UINT32*)((UINT32)&RcvFis+0x60), (sizeof(UnknownFis)/4), 0, 0);
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	clearDmaBuffers( )							*
 *      Initialize Receive FIS space						*
 * ---------------------------------------------------------------------------- */
void clearDmaBuffers(void)
{
	// Clear PRD Data Buffer Memory
	initMemory((UINT32 *)prdTableDataBuff, (LISTLENGTH*PRDLENGTH*DATABUFFERLEN/4), 0, 0);
}

/* ------------------------------------------------------------------------------------	*
 *											*
 *    invokeHBAReset( )									*
 *    Reset HBA										*
 *    HBA Reset will not affect the following Registers settings of PxFB and PxCLB	*
 *    regs and HwInit fields of Port Registers are not affected.			*
 *    To Do: Check if the Global Registers are affected. Spec mentions not affected.	*
 *											*
 *	Note: COMRESET OOB will not be sent to attached Device because Freon supports	*
 *	  Staggered Spinup capability and P0CMD.SUD is cleared to Zero when HBA Reset	*
 *	  takes place. Software needs to invoke this if needed.				*
 *											*
 *	Most likely user want to ensure HBA comes up in its default operation state	*
 *	  or has hung and are unable to idle the port when needing to perform an HBA	*
 *	  reset. Regardless, there is no need to attempt to idle the HBA from running.	*
 * ------------------------------------------------------------------------------------	*/
void invokeHBAReset( ) 
{
	sataRegs->GHC |= (1 << AHCI_GHC_HR_SHIFT);  // HBA Global Reset
	while((sataRegs->GHC & AHCI_GHC_HR) != 0)   // Max Spec time is 1 Second for Reset to complete.
		DM814x_usecDelay(1000000);
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	performFirmwareInit( )							*
 *      Initialize Firmware capabilities fields and configure SATA controller	*
 * ---------------------------------------------------------------------------- */
void performFirmwareInit( )
{
	/* Firmware Initialization*/
	sataRegs->CAP |= ((swCtrlFeatures.capSMPS << 28) |  // Make sure you perform a Single Write in one operation of all HwInit Fields
	                  (swCtrlFeatures.capSSS  << 27)
	                 );

	// Configure PI[31:0]
	sataRegs->PI  |=  (swCtrlFeatures.piPi  << 0);

	// Configure P0CMD[ESP,CPD,MPSP,HPCP=21,20,19,18]
	sataRegs->P0CMD |= ((swCtrlFeatures.p0cmdEsp << 21) |
                        (swCtrlFeatures.p0cmdCpd << 20) |
                        (swCtrlFeatures.p0cmdMpsp << 19)|
                        (swCtrlFeatures.p0cmdHpcp << 18)
                       );

	/* Software Initialization */
	// Port PHY Control Register
#if(0)
	// For AM1808
	sataRegs->P0PHYCR =  8		<< 0	|  // MPY		
			     0		<< 4	|  // LB		
			     1		<< 6	|  // LOS		
			     0 		<< 7	|  // RXINVPAIR	
			     0		<< 8	|  // RXTERM	
			     4		<< 10	|  // RXCDR		
			     1		<< 13	|  // RXEQ		
			     0 		<< 17	|  // TXINVPAIR	
			     0		<< 18	|  // TXCM		
			     3 		<< 19	|  // TXSWING	
			     0		<< 22	|  // TXDE		
			     0 		<< 30	|  // OVERRIDE	
			     1		<< 31;     // ENPLL	
#endif

	// For Netra
	sataRegs->P0PHYCR =  0      << 27   |  // TXDE
	                     3      << 23   |  // TXSWING
	                     0      << 22   |  // TXCM
	                     0      << 21   |  // TINVPAIR
	                     1      << 20   |  // RXENOC
	                     1      << 16   |  // RXEQ
	                     4      << 13   |  // RXCDR
	                     1      << 12   |  // LOS
	                     0      << 10   |  // LOOPBACK
	                     0      <<  9   |  // RXINVPAIR
	                     0      <<  7   |  // CLKBYP
	                     0      <<  5   |  // LB
	                     8      <<  1   |  // MPY
	                     1      <<  0;     // ENPLL			 
						 
	initBaseAddresses();          // Initialize Command List (P0CLB) and Receive FIS (P0FS)
	// Configure Line Speed.
	setSataSpeed(DESIRED_SPEED);  // GOASFASTASDEVICE=0,GEN1=1,GEN2=2
	enableRcvFis();	              // Enable Receive DMA
	// The below are general Semaphores/Flags used and want to make sure they are initialized.
	// 28 Bit LBA Address of Device. 0xFFFFFFFF is used by test S/W to indicate that it is not initialized
	Dev28bitLbaAddress = 0xFFFFFFFF; 
	PioCmd = 0;      // Semaphore is used to either invoke IDENTIFY DEVICE Cmd or READ SECTORS Cmd.
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	cfgDmaSetting( )							*
 *      Configure DMA control register						*
 * ---------------------------------------------------------------------------- */
void cfgDmaSetting( ) 
{
	// Port DMA Control Register
	sataRegs->P0DMACR   |= ((DMA_BURST_LENGTH << AHCI_P0DMACR_TXTS_SHIFT)       |  // Receive VBUSP Burst limit
	                        (DMA_BURST_LENGTH << AHCI_P0DMACR_RXTS_SHIFT)       |  // Transmit VBUSP Burst limit 
	                        (DMA_TRANACTION_LENGTH << AHCI_P0DMACR_TXABL_SHIFT) |  // Receive Size
	                        (DMA_TRANACTION_LENGTH << AHCI_P0DMACR_RXABL_SHIFT)    // Transmit Size
	                       );                                                      // Values assigned in sata.h
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	initIntAndClearFlags( )							*
 *      Initialize interrupts and clear interrupt and error flags		*
 * ---------------------------------------------------------------------------- */
void initIntAndClearFlags( ) 
{
	// Make sure Interrupt is disabled (Disable at Port Level followed by Global Level).
	// Clear Interrupt at Port Level
	enableDisableInt(PORTint, DISABLE, 0xFFFFFFFF); // clearInt(int type, intState, specificField)
							// int type=GLOBALint or PORTint
							// intState=DISABLE or ENABLE
							// specificField=bit field to Enable or Disable
							//  is used for the RWC feature for PORTint
	// Disable interrrupt at Global Level
	enableDisableInt(GLOBALint, DISABLE, 0);    	// clearInt(intType, intState fields2clr)
							// int type=GLOBALint or PORTint
							// intState=DISABLE or ENABLE
							// fields2clr=dontcare for GLOBALint
							//  is used for the RWC feature for PORTint

	// Need to clear interrupts at Port Level followed by Global Level.
	// Ensure all pending Port Error and Status are cleared.
	clearIntOrErrorDiag(ERRORFIELDS, sataRegs->P0SERR);	// Clear P0SERR Register
	// Ensure all pending Port Interrupts and The Single Global Interrupt are cleared.
	clearIntOrErrorDiag(INTFIELDS, sataRegs->P0IS);     // Clear P0IS and IS Regs
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	initBaseAddresses( )							*
 *      Initialize Command List and Receive FIS					*
 * ---------------------------------------------------------------------------- */
void initBaseAddresses( )
{
	sataRegs->P0CLB=(unsigned int)CmdLists;  // Set Command List Base Address
	sataRegs->P0FB=(unsigned int)&RcvFis;    // Set Port FIS Base Address Register
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	setSataSpeed( )								*
 *      Configure Max Speed Allowed						*
 * ---------------------------------------------------------------------------- */
void setSataSpeed(UINT8 iSpeed)
{
	sataRegs->P0SCTL &= ~(AHCI_PxSCTL_PxSSTS_SPD);
	sataRegs->P0SCTL |= (iSpeed << AHCI_PxSCTL_PxSSTS_SPD_SHIFT);  // Set Speed Allowed
	DM814x_usecDelay(5000);
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	enableRcvFis( )								*
 *      Enable Receive FIS							*
 * ---------------------------------------------------------------------------- */
void enableRcvFis( )
{
	sataRegs->P0CMD |= AHCI_PxCMD_FRE;	                        // FIS Receive Enable.
	while((sataRegs->P0CMD & AHCI_PxCMD_FR) != AHCI_PxCMD_FR);  // Wait for FIS Receive Running bit
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	enableDisableInt( )							*
 *      Enable or Disable Interrupts						*
 * ---------------------------------------------------------------------------- */
void enableDisableInt(UINT8 intType, UINT8 enableDisable, UINT32 intPosition)
{
	switch (intType) 
	{
		case GLOBALint:                         // Global
			if(enableDisable==ENABLE)
				sataRegs->GHC |= (1<<1);        // Enable
			else
				sataRegs->GHC &= ~(1<<1);       // Disable
			break;
		case PORTint: 
			if(enableDisable==ENABLE)           // Port
				sataRegs->P0IE |= intPosition;  // Enable
			else
				sataRegs->P0IE &= ~intPosition; // Disable
			break;
		default:
			break;
	}
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	clearIntOrErrorDiag( )							*
 *      Clear Interrupt or Error flags						*
 * ---------------------------------------------------------------------------- */
void clearIntOrErrorDiag(UINT8 intOrErrorDiagType, UINT32 intErrorPosition)
{
	switch (intOrErrorDiagType) 
	{
		case ERRORFIELDS:  // Errors
			sataRegs->P0SERR |= intErrorPosition; // Clear bits in Port Serial ATA Error Register
			break;

		case INTFIELDS:    // Interrupts
			sataRegs->P0IS |= intErrorPosition;  // Clear bits in Port Interrupt Status Register
			sataRegs->IS |= 0x1;                 // Clear Port 0 Interrupt
			break;
		default:
			break;
	}
}

/* ---------------------------------------------------------------------------- *
 *										*
 *	spinUpDeviceAndWaitForInitToComplete( )					*
 *      Spin up and negotiate speed with device					*
 * ---------------------------------------------------------------------------- */
char spinUpDeviceAndWaitForInitToComplete(void)
{
	UINT32 timeout;

	
	//  Following Configuration is not allowed.
	//      [P0SCTL.DET, P0CMD.SUD] = [1,1] NOT Allowed.
	if((sataRegs->P0SCTL & AHCI_PxSCTL_PxSSTS_DET) != 0)  // Make sure that the HBA is in a Listen Mode prior to Spinning Up Device
		sataRegs->P0SCTL &= ~(0xf << AHCI_PxSCTL_PxSSTS_DET_SHIFT);
	sataRegs->P0SERR |= 0x04000000;  // Clear P0SERR.DIAG.X (RWC bit field) so that the P0TFD is updated by HBA
	/* Spin Up Device */
	sataRegs->P0CMD |= (1 << AHCI_PxCMD_SUD_SHIFT);
	// Wait for Device Detection or/and Speed Negotiation to take place and finish
	timeout = 0;
	while ((sataRegs->P0SSTS & AHCI_PxSCTL_PxSSTS_DET) !=0x3)
	{
		DM814x_usecDelay(500000);
		timeout++;
		if(timeout > 10)
		    return 1;
	}  // Device sends its status and and default Task file regs content (signature)

    while ((sataRegs->P0TFD & AHCI_PxTFD_STS_BSY_DRQ_ERR) != 0);  // Look for Device ready status.
	// Make sure that the expected Device signature is received.
	timeout = 0;
	while ((sataRegs->P0SIG & 0xFFFF) != AHCI_P0SIG_SIG_ATA_DEV_GOOD_STAT) // LBAhigh:LBAmid:LBAlow:SECcnt
	{
		DM814x_usecDelay(500000);		
		timeout++;
		if(timeout > 10)
		    return 1;
	}
	return 0;
}

/* ---------------------------------------------------------------------------- *
 *	setupCfisEntriesForDataRdWr(*CmdListNum, readOrWrite, xferType)		*
 *	    Initializes the Command Header					*
 *										*
 *    Other parts of the Command List Structure, with the exception of Word 0 	*
 *    for the Command Slots[0] and [1], are already initialized when invoking 	*
 *    sata_init_and_spin_up() function via associateMem2HBA() function.		*
 * ---------------------------------------------------------------------------- */
char setupCfisEntriesForDataRdWr(CmdListHeader *CmdListNum, dataXferDir readOrWrite, xferProtocol xferType) 
{
	/* Configure Word 0 of Command List */
	CmdListNum->DW0.CmdLen=5;   // This is the length of H2D FIS. This might need changing
							    //   based on the Command issued to Device. Need to Check.
    CmdListNum->DW0.Atapi=0;    // Command is destined to HDD Like Device.
	CmdListNum->DW0.Prefetch=1; // Doesn't hurt prefetching so do it.
								// WARNING: Do Not Prefetch if using:
								//          => Command Queuing
								//          => Port Multiplier
	CmdListNum->DW0.Reset=0;    // This is normally set to Zero unless a Soft Reset is required.
	CmdListNum->DW0.Bist=0;     // This is for entering test mode and should be cleared for normal operation.
	CmdListNum->DW0.Rok=0;      // For Normal operation require to Clear this bit so P0TFD and P0CI are modified by HBA as appropriate.
								//  Rok should be set for S/W Reset Command.
	CmdListNum->DW0.Pmp=0x0;    // Used only if an external Port Multiplier is attached and selects the Port of the Port Multiplier.
	/* The above DW0 fields usually would not change for Normal operation. */

	if (readOrWrite == DATA_DIR_WR) // The Write setting here is based on the Data FIS direction.
		CmdListNum->DW0.Write=1;    //  Write=1/0=>Write/Read;
	else if (readOrWrite == DATA_DIR_RD) // The Write setting here is based on the Data FIS direction.
		CmdListNum->DW0.Write=0;    //  Write=1/0=>Write/Read;
	else return(1);

	CmdListNum->DW0.Prdtl=PRDLENGTH;    // Need to update this when using DMA for Data transfer.

    /* ---------------------------------------------------------------------------- *
     *										    *
     *	Initializing the Command FIS (H2D FIS)					    *
     *      Cmd FIS is made of a 20 Bytes size FIS.				    *
     *      FIS Fieds are Initialized into the alloted data buffers.		    *
     *      Strucutre type 'cmdFis' holds 8 of the 20 bytes of the FIS. These seven *
     *        elements of the cfis are good enough for majority of command building *
     *        tasks.								    *
     *      If need to access other cfis members, need to access the cfis member    *
     *        directly. Example of how to access the FIS Type, cfis Byte0.	    *
     *        : CmdTable[n].cfis.DW0.B0FisType=0x27;				    *
     *      FIS Type is hard coded within buildCmdFis function and Reserved 	    *
     *        Locations are also cleared to Zeros, so no need of iniitalizing 	    *
     *        this bytes is necessary here.					    *
     * ---------------------------------------------------------------------------- */

	// Make sure the Device 28 Bit LBA Address is initialize prior to calling this function.
	if (Dev28bitLbaAddress == 0xFFFFFFFF)
		return(1);

	myCmdFis.cfisByte1             = CMDFIS_BYTE1_C_IS_CMD_UPDATE;	// Bit7 of Byte1 is set for Command Wr and Cleared for Control Wr
	//myCmdFis.cfisByte1             = CMDFIS_BYTE1_C_IS_CTRL_UPDATE; // Bit7 of Byte1 is Cleared for Command Control Wr.

	if (readOrWrite == DATA_DIR_WR) 
	{
		if (xferType == DMA_PROTOCOL) 
		{
			myCmdFis.cfisCmd               = ATA_CMD_WRITE_DMA;  // Uses 28-Bit Addressing.
			//myCmdFis.cfisCmd               = ATA_CMD_WRITE_DMA_EXT; // Uses 48-Bit Addressing.
		} 
		else 
			myCmdFis.cfisCmd               = ATA_CMD_WRITE_SECTOR; // PIO Write Command: Uses 28-Bit Addressing
	}
	else
	{
		if (xferType == DMA_PROTOCOL) 
		{
			myCmdFis.cfisCmd               = ATA_CMD_READ_DMA;  // Uses 28-Bit Addressing.
			//myCmdFis.cfisCmd               = ATA_CMD_READ_DMA_EXT; // Uses 48-Bit Addressing.
		} 
		else 
		{
			if (PioCmd == ATA_CMD_IDENTIFY_DEVICE)
				myCmdFis.cfisCmd               = ATA_CMD_IDENTIFY_DEVICE; // PIO Read Command: Uses 28-Bit Addressing.
			else
				myCmdFis.cfisCmd               = ATA_CMD_READ_SECTOR;     // PIO Read Command: Uses 28-Bit Addressing.
		}
	}

	myCmdFis.cfisFeature           = 0x00;
	myCmdFis.cfisDw1SecNumLbaLow   = (UINT8) Dev28bitLbaAddress;
	myCmdFis.cfisDw1CylLowLbaMid   = (UINT8)(Dev28bitLbaAddress>>8);
	myCmdFis.cfisDw1CylHighLbahigh = (UINT8)(Dev28bitLbaAddress>>16);
	myCmdFis.cfisDw1Dev            = ( DEVICE_REG_USE_LBA_ADDRESSING |
	                                  (UINT8)(Dev28bitLbaAddress>>24)
	                                 );
	myCmdFis.cfisDw3SecCnt         = (PRDLENGTH*DATABUFFERLEN)/512;
	myCmdFis.cfisDw3Ctrl           = 0x00;

	// The below require to be initialized at least once and can be ignored especially if
	//  not using 48-Bit Addressing. If use 48-Bit Addressing, then require constant
	//  maintenance prior to invoking a command.
	myCmdFis.cfisDw2SecNumLbaLowExp=0x00;
	myCmdFis.cfisDw2CylLowLbaMidExp=0x00;
	myCmdFis.cfisDw2CylHighLbahighExp=0x00;
	myCmdFis.cfisDw2FeatureExp=0x00;
	myCmdFis.cfisDw3SecCntExp=0x00;

	// Invalidate for future use.
	Dev28bitLbaAddress = 0xFFFFFFFF;

	return(0);
}

/* ---------------------------------------------------------------------------- *
 *	buildCmdFis(*CmdSlotNum)						*
 *	    Assign values of Command FIS to associated command slot		*
 *										*
 * ---------------------------------------------------------------------------- */
/*
COMMAND FIS
-----------
    +----------+----------+--------------+-----------+
 DW0|  FEATURE | COMMAND  | c r r r port |FISTYPE 27h|
    +----------+----------+--------------+-----------+
 DW1|  DEVICE  | LBA HIGH |  LBA MID     |  LBA LOW  |
    +----------+----------+--------------+-----------+
 DW2|FETURESexp|LBAHIGHexp|  LBAMIDexp   | LBALOWexp |
    +----------+----------+--------------+-----------+
 DW3| CONTROL  | RESERVED | SEC CNTexp   |  SEC CNT  |
    +----------+----------+--------------+-----------+
 DW4| RESERVED | RESERVED |  RESERVED    | RESERVED  |
    +----------+----------+--------------+-----------+
*/
void buildCmdFis(CommandTable *CmdSlotNum)
{
	CmdSlotNum->cfis.DW0.B0FisType=0x27;
	CmdSlotNum->cfis.DW0.BYTE1=myCmdFis.cfisByte1;               //Make Sure the 'C' bit field is correctly set or cleared.
	CmdSlotNum->cfis.DW0.B2Cmd=myCmdFis.cfisCmd;
	CmdSlotNum->cfis.DW0.B3Feature=myCmdFis.cfisFeature;

	CmdSlotNum->cfis.DW1.B0LbaLow=myCmdFis.cfisDw1SecNumLbaLow;
	CmdSlotNum->cfis.DW1.B1LbaMid=myCmdFis.cfisDw1CylLowLbaMid;
	CmdSlotNum->cfis.DW1.B2LbaHigh=myCmdFis.cfisDw1CylHighLbahigh;
	CmdSlotNum->cfis.DW1.B3Device=myCmdFis.cfisDw1Dev;           //Make Sure 48-Bit or 28-Bit Addressing is indicated here.


	CmdSlotNum->cfis.DW2.B0LbaLowExp=myCmdFis.cfisDw2SecNumLbaLowExp;    //0x0;
	CmdSlotNum->cfis.DW2.B1LbaMidExp=myCmdFis.cfisDw2CylLowLbaMidExp;    //0x0;
	CmdSlotNum->cfis.DW2.B2LbaHighExp=myCmdFis.cfisDw2CylHighLbahighExp; //0x0;
	CmdSlotNum->cfis.DW2.B3FeatureExp=myCmdFis.cfisDw2FeatureExp;        //0x0;

	CmdSlotNum->cfis.DW3.B0SecCnt=myCmdFis.cfisDw3SecCnt;
	CmdSlotNum->cfis.DW3.B1SecCntExp=myCmdFis.cfisDw3SecCntExp;          //0x0;
	CmdSlotNum->cfis.DW3.B2Rsv=0x0;
	CmdSlotNum->cfis.DW3.B3Control=myCmdFis.cfisDw3Ctrl;

	CmdSlotNum->cfis.DW4.DWResv=0x0;
}

/* ---------------------------------------------------------------------------- *
 *	associtateCmdSlotWithCmdTable(cmdSlot)					*
 *	    Associate command list slot to command table			*
 *										*
 * ---------------------------------------------------------------------------- */
void associtateCmdSlotWithCmdTable(UINT16 cmdSlot)
{
	CmdLists[cmdSlot].DW2.CmdTableAddLow = ((unsigned int)&CmdTable[cmdSlot] & 0xFFFFFF80);
	CmdLists[cmdSlot].DW3.CmdTableAddHigh=0x0;
}

/* ---------------------------------------------------------------------------- *
 *	associtatePrdsWithCmdTable(cmdSlot)					*
 *	    Associate PRD Table Data Buffer to command table			*
 *										*
 * ---------------------------------------------------------------------------- */
void associtatePrdsWithCmdTable(UINT16 cmdSlot)
{
	UINT16 prdLength;
	for (prdLength=0; prdLength<PRDLENGTH; prdLength++)
	{
		CmdTable[cmdSlot].prdTable[prdLength].DW0.DbaLow=(unsigned int)&prdTableDataBuff[cmdSlot][prdLength];
		CmdTable[cmdSlot].prdTable[prdLength].DW1.DbaHigh=0x0;
		CmdTable[cmdSlot].prdTable[prdLength].DW3.DataBC=DATABUFFERLEN-1;
	}
}

/* ---------------------------------------------------------------------------- *
 *	associateSysMem2Hba(cmdSlot)						*
 *	    Assign PRD and Command Table to Command list			*
 *										*
 * ---------------------------------------------------------------------------- */
void associateSysMem2Hba(UINT16 cmdSlot)
{
	associtateCmdSlotWithCmdTable(cmdSlot);	// Assign Sys Mem allocated for Cmd Table to Cmd List Slot
	associtatePrdsWithCmdTable(cmdSlot); 	// Assign PRD info to Cmd Table
}

/* ---------------------------------------------------------------------------- *
 *	getRegStatus(* pIntReg, intFieldMask))					*
 *	    Check intFieldMask of pIntReg					*
 *										*
 * ---------------------------------------------------------------------------- */
UINT32 getRegStatus(UINT32 * pIntReg, UINT32 intFieldMask)
{
	return(*pIntReg & intFieldMask);
}

/* ---------------------------------------------------------------------------- *
 *	submitCmd( commandType, commandSlot) 					*
 *	    Issue Command in commandSlot slot 					*
 *										*
 * ---------------------------------------------------------------------------- */
char submitCmd(UINT8 commandType, UINT8 commandSlot) 
{
	// Make sure both the Command List and Receive FIS DMAs are eanbled and running prior to submiting command
	if ((sataRegs->P0CMD & (AHCI_PxCMD_FRE | AHCI_PxCMD_FRE | AHCI_PxCMD_CR | AHCI_PxCMD_ST)) !=
	                         (AHCI_PxCMD_FRE | AHCI_PxCMD_FRE | AHCI_PxCMD_CR | AHCI_PxCMD_ST))
		return(1);

	switch (commandType) 
	{
			case NON_QUEUED_CMD:
				sataRegs->P0CI |= (0x1 << commandSlot);  // Issue Command
				break;

			case QUEUED_CMD:
				sataRegs->P0SACT |= (0x1 << commandSlot); // Queued Command
				sataRegs->P0CI |= (0x1 << commandSlot);;  // Issue Command
				break;

			default:
				break;
	}
	return(0);
}

/* ---------------------------------------------------------------------------- *
 *	startCmdListProcessing( ) 						*
 *	    Enable the command list to start processing commands		*
 *										*
 * ---------------------------------------------------------------------------- */
char startCmdListProcessing( )
{
	// Make sure device is detected and HBA established communications
	while ((sataRegs->P0SSTS & AHCI_PxSCTL_PxSSTS_DET) !=0x3);
	// Clear P0SERR.DIAG.X (RWC bit field) so that the P0TFD is updated by HBA
	sataRegs->P0SERR |= 0x04000000;       // Write 1 to clear
	if (sataRegs->P0CMD & AHCI_PxCMD_CR)  // Make sure the Command List is not Running
		return(1);
	while ((sataRegs->P0TFD & AHCI_PxTFD_STS_BSY_DRQ_ERR) != 0);  //  Look for Device ready status
	if ((sataRegs->P0CMD & (AHCI_PxCMD_FRE | AHCI_PxCMD_FRE)) !=  // Make sure the Receive FIS DMA is running
	                         (AHCI_PxCMD_FRE | AHCI_PxCMD_FRE))
		return(1);
	sataRegs->P0CMD |= AHCI_PxCMD_ST;              // Enable the Cmd List DMA Engine
	while ((sataRegs->P0CMD & AHCI_PxCMD_CR) == 0) // Wait for the Command List DMA Engine to start
		DM814x_usecDelay(1000);	

	return(0);
}

/* ------------------------------------------------------------------------------------------------	*
 *	performDmaWrite(Addr)										*
 *	    Start DMA Write to LBA address Addr of device						*
 *													*
 *	When performing the Non-Queued Command "Write DMA" the following captures the FISes that are	*
 *	  communicated between Host (HBA) and Device (SpeedBridge or SATA Drive).			*
 *	HBA ==> Device 		H2D FIS (Command FIS that has the "Write DMA" Command within H2D.Command*
 *						field).							*
 *	Device ==> HBA 		DMA Activate FIS (Device notifies HBA that it's ready to receive data).	*
 *	HBA ==> Device 		Data FIS (Actual Data Requested and pointed by the DMA/PRD Contents within*
 *						Command Table).						*
 *	Device ==> HBA 		D2H FIS (Completion Status from Device)					*
 *	If need to capture Interrupt (just have only the Flag set) at:					*
 *	  Global Level (IS Register), need to enable P0IE.Interrupt field.				*
 *	Port Level (No Enable bit needs to be set. P0IS will be set if D2H FIS DW0.Byte1.bit6 is set).	*
 *													*
 * ------------------------------------------------------------------------------------------------ 	*/
void performDmaWrite(UINT32 Addr) 
{
	cmdSlot2Use=0;                     // Write data to HDD from slot 0 of prdTableDataBuff
	associateSysMem2Hba(cmdSlot2Use);  // Associate Sys Memory to CmdList and PRDs to Cmd Table
	Dev28bitLbaAddress =  Addr;        // 28 bit LBA Address (Start Address for R/W Transfer)
	// If problem exist when setuping CmdList, stay here. If not continue
	setupCfisEntriesForDataRdWr(&CmdLists[cmdSlot2Use], DATA_DIR_WR, DMA_PROTOCOL); 
	// Usage:setupCfisEntriesForDataRdWr(CmdListHeader *, DataDir, xferProtocol)
	// DataDir = DATA_DIR_RD or DATA_DIR_WR, xferProtocol = PIO/DMA_PROTOCOL
	
	buildCmdFis(&CmdTable[cmdSlot2Use]);     // Write cfis Data
	startCmdListProcessing();                // Stay here if unable to start processing command list
	submitCmd (NON_QUEUED_CMD, cmdSlot2Use); // Submit Write command
											 //   Usage: CommandType (QUEUED(NON_QUEUED)_CMD,Command Slot);
    while(sataRegs->IS == 0);                // Stay here until an interrupt is received. -> Poll interrupt Status
	                                         // If P0IS.DPS is set, A PRD with the I bit set has transferred all of its data.
	while(getRegStatus((UINT32*)&sataRegs->P0CI, (1<<cmdSlot2Use)) == (1<<cmdSlot2Use)); //	Wait Until P0CI[cmdSlot] to be cleared by HBA
	clearIntOrErrorDiag(INTFIELDS, sataRegs->P0IS);  // Clears both P0IS and IS register. Only Clears RWC bit fields
}

/* ------------------------------------------------------------------------------------------------------------	*
 *	performDmaRead(Addr)											*
 *	    Start DMA Read from LBA address Addr of device							*
 *														*
 *	When performing the Non-Queued Command "Read DMA" the following captures the FISes that are		*
 *	  communicated between Host (HBA) and Device (SpeedBridge or SATA Drive).				*
 *	HBA ==> Device 		H2D FIS (Command FIS that has the "Read DMA" Command within H2D.Command		*
 *						field).								*
 *	Device ==> HBA 		Data FIS (Actual Data Requested and pointed by the DMA/PRD Contents within	*
 *						Command Table).							*
 *	Device ==> HBA 		D2H FIS (Completion Status from Device)						*
 *	If need to capture Interrupt (just have only the Flag set) at:						*
 *	  Global Level (IS Register), need to enable P0IE.Interrupt field.					*
 *	Port Level (No Enable bit needs to be set. P0IS will be set if D2H FIS DW0.Byte1.bit6 is set).		*
 *														*
 * -----------------------------------------------------------------------------------------------------------	*/
void performDmaRead(UINT32 Addr) 
{
	cmdSlot2Use = 1;                  // Store data read from HDD in slot 1 of prdTableDataBuff
	associateSysMem2Hba(cmdSlot2Use); // Associate Sys Memory to CmdList and PRDs to Cmd Table
	Dev28bitLbaAddress = Addr;        // 28 bit LBA Address (Start Address for R/W Transfer)
	setupCfisEntriesForDataRdWr(&CmdLists[cmdSlot2Use], DATA_DIR_RD, DMA_PROTOCOL); 
	// Usage: setupCfisEntriesForDataRdWr(CmdListHeader *, DataDir, xferProtocol)
	// DataDir = DATA_DIR_RD or DATA_DIR_WR, xferProtocol = PIO/DMA_PROTOCOL

	buildCmdFis(&CmdTable[cmdSlot2Use]);     // Write cfis Data
	startCmdListProcessing();                // Stay here if unable to start processing command list.
	submitCmd (NON_QUEUED_CMD, cmdSlot2Use); // Submit Write command
											 //   Usage: CommandType (QUEUED(NON_QUEUED)_CMD, Command Slot);
    while(sataRegs->IS == 0);                // Stay here until an interrupt is received. -> polling
                                           	 // If P0IS.DPS is set, A PRD with the I bit set has transferred all of its data.
	while(getRegStatus((UINT32*)&sataRegs->P0CI, (1<<cmdSlot2Use)) == (1<<cmdSlot2Use)); // Wait Until P0CI[ChHeader] to be cleared by HBA
	clearIntOrErrorDiag(INTFIELDS, sataRegs->P0IS);  // Clears both P0IS and IS register. Only Clears RWC bit fields.
}

